/* ------------------------------------------------------------------------ */
/*  @@ Source Documentation                            *** TC Version ***   */
/*                                                                          */
/*  Copyright (c) Creative Technology Pte Ltd, 1991. All rights reserved.   */
/*                                                                          */
/*   TITLE       : DEMOVMR.C                                                */
/*                                                                          */
/*   DESCRIPTION :                                                          */
/*       This program demostrates how to perform voice recording using the  */
/*       CT-VOICE.DRV driver. The voice recording is using the Conventional */
/*       memory method. The recording can be terminated by pressing ESC.    */
/*                                                                          */
/*       The program checks BLASTER environment for the Card settings.      */
/*       It also performs test base on BLASTER environment settings to      */
/*       ensure they are tally with the hardware settings on the Card.      */
/*                                                                          */
/*       Note that the program included the module LOADDRV.C to load        */
/*       the loadable CT-VOICE.DRV into memory.                             */
/*                                                                          */
/* ------------------------------------------------------------------------ */


#include  <io.h>
#include  <string.h>
#include  <stdlib.h>
#include  <dos.h>
#include  <bios.h>
#include  <stdio.h>
#include  <fcntl.h>

#include  <sbc.h>
#include  <sbcvoice.h>

#define   FPSEG(fp) (*((unsigned far *)&(fp)+1))
#define   FPOFF(fp) (*((unsigned far *)&(fp)))

#include  "loaddrv.c"


/* Function Prototypes */
SaveVoiceFile(char *,char far *) ;


main()
{
    extern  char far * near voice_drv;


    /* Retrieve the BLASTER environment settings */
    if ( ! GetEnvSetting() )
    {
        if (sbc_check_card() & 4)
        {
            if (sbc_test_int())
            {
                if (sbc_test_dma() >= 0)
                {
                    if ((voice_drv = LoadDriver("CT-VOICE.DRV")) != 0)
                    {
                        if (!ctvm_init())
                        {
                            ctvm_speaker(0) ;

                            RecordVoice("TEMP.VOC");

                            ctvm_terminate() ;
                        }
                    }
                }
                else
                    printf("Error on DMA channel.\n");
            }
            else
                printf("Error on interrupt.\n");
        }
        else
            printf("Sound Blaster Card not found or wrong I/O settings.\n") ;
    }
    else
        printf("BLASTER environment not set or incomplete or invalid.\n");
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*   RecordVoice (char *szFilename)                                         */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Record voice into a file with filename specified.                  */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       szFileName :- File to be recorded.                                 */
/*                                                                          */
/*   EXIT:                                                                  */
/*       None                                                               */
/*                                                                          */
/* ------------------------------------------------------------------------ */

RecordVoice (char *szFilename)
{
    unsigned  wSeg ;
    char      far *lpVoiceBuf ;
    long      lBufSize = 0x20000L;


    if (allocmem((unsigned)((lBufSize+15) >> 4),&wSeg) == -1)
    {
        FPSEG(lpVoiceBuf) = wSeg ;
        FPOFF(lpVoiceBuf) = 0 ;

        if (Recording(lpVoiceBuf,lBufSize))
            SaveVoiceFile(szFilename,lpVoiceBuf) ;

        freemem(FPSEG(lpVoiceBuf)) ;
     }
     else
        printf("Memory allocation error.\n");
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*   Recording (char far *lpBuf, long lBufSize)                             */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Start recording voice.                                             */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       lpBuf :- buffer for voice recording.                               */
/*       lBufSize :- buffer size.                                           */
/*                                                                          */
/*   EXIT:                                                                  */
/*       Non-zero if successful, else returns 0.                            */
/*                                                                          */
/* ------------------------------------------------------------------------ */

#pragma loop_opt(off)
Recording (char far *lpBuf,long lBufSize)
{
    int    RetVal = 0 ;


    ctvm_speaker(0) ;

    if (ctvm_input(lpBuf,lBufSize,8000) == NO_ERROR)
    {
        RetVal = 1 ;
        printf("\nStart recording, press ESC key to terminate .....");

        while (ct_voice_status)
        {
            if (bioskey(1))
            {
                 /* check for escape key */
                if (bioskey(0) == 0x11b)
                    ctvm_stop() ;
            }
        }
    }


    return(RetVal) ;
}
#pragma loop_opt()


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*   SaveVoiceFile (char *szFilename, char far *lpBuf)                      */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Save recorded voice from memory to file.                           */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       szFilename :- file name to be saved to.                            */
/*       lpBuf :- recorded voice buffer.                                    */
/*                                                                          */
/*   EXIT:                                                                  */
/*       None                                                               */
/*                                                                          */
/* ------------------------------------------------------------------------ */

SaveVoiceFile (char *szFilename, char far *lpBuf)
{
    int     Handle ;
    long    lVoiceSize ;
    VOCHDR  stHeader ;


    strcpy(stHeader.id,"Creative Voice File\x0\0x1A") ;
    stHeader.voice_offset = sizeof(VOCHDR) ;
    stHeader.version = 0x10a ;
    stHeader.check_code = ~stHeader.version + 0x1234 ;

    if ( (Handle=_creat(szFilename,0)) == -1 )
        printf("Create %s error.\n",szFilename) ;
    else
    {
        /* write voice header */
        if (WriteToFile(Handle,(char far*)&stHeader,(long)sizeof(VOCHDR)))
        {
            /* get the recorded data block size */
            lVoiceSize = *(lpBuf+3) ;
            lVoiceSize <<= 16 ;
            lVoiceSize += *(unsigned far *)(lpBuf+1) ;

		    /*	include the block type, block length  */
            /*  and terminator block type             */

            lVoiceSize += 5 ;

            WriteToFile(Handle,lpBuf,lVoiceSize) ;
        }

        _close(Handle) ;
    }
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*   WriteToFile (int Handle, char far *lpBuf, long lSize)                  */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Write data from buffer to file.                                    */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       Handle :- File handle where data to be written to.                 */
/*       lpBuf :- buffer to be written to file.                             */
/*       lSize :- Size to be written to file.                               */
/*                                                                          */
/*   EXIT:                                                                  */
/*       Non-Zero if successful, else returns 0.                            */
/*                                                                          */
/* ------------------------------------------------------------------------ */

WriteToFile (int Handle,char far *lpBuf,long lSize)
{
    int         RetVal = 1 ;
    unsigned    wByteToWrite, wByteWritten ;


    while (lSize)
    {
        wByteToWrite = 0x8000 ;

        if (lSize < 0x8000)
            wByteToWrite = (int)lSize ;

        if ( DosWrite(Handle,lpBuf,wByteToWrite,&wByteWritten) == 0 )
        {
            printf("Write file error.\n") ;
            RetVal = 0 ;
            break ;
        }
        else
        {
            if (wByteWritten != wByteToWrite)
            {
                printf("Disk full.\n") ;
                RetVal = 0 ;
                break ;
            }
            else
            {
                if (FPOFF(lpBuf) < 0x8000)
                    FPOFF(lpBuf) += wByteWritten ;
                else
                {
                    FPOFF(lpBuf) += wByteWritten ;
                    FPSEG(lpBuf) += 0x1000 ;
                }

                lSize -= wByteWritten ;
            }
        }
    }


    return(RetVal) ;
}


/* ------------------------------------------------------------------------ */
/*  @@ Usage                                                                */
/*                                                                          */
/*   DosWrite (int Handle, char far *Buffer,                                */
/*             unsigned wLen, unsigned *lpByteRead)                         */
/*                                                                          */
/*   DESCRIPTION:                                                           */
/*       Write data from file using DOS interrupt 0x21 function 0x40.       */
/*                                                                          */
/*   ENTRY:                                                                 */
/*       Handle :- File handle to read.                                     */
/*       Buffer :- Buffer to write to.                                      */
/*       wLen   :- Number of byte to read.                                  */
/*       wByteWritten :- pointer to number of byte actually written.        */
/*                                                                          */
/*   EXIT:                                                                  */
/*       Byte written if successful, else returns 0.                        */
/*                                                                          */
/* ------------------------------------------------------------------------ */

DosWrite (int Handle, char far *Buffer, unsigned wLen, unsigned *wByteWritten)
{
    union REGS regs;
    struct SREGS segregs;


    regs.h.ah = 0x40 ;
    regs.x.bx = Handle;
    regs.x.dx = FPOFF(Buffer);
    regs.x.cx = wLen;
    segregs.ds = FPSEG(Buffer);

    intdosx(&regs, &regs, &segregs);

    if(regs.x.cflag)    /* error */
        *wByteWritten = 0;
    else
        *wByteWritten = regs.x.ax ;


    return(*wByteWritten);
}

